//*************************************************************************************************
//
//	Description:
//		dashboard.fx - Specialisation of the basic object shader which supports lit dashboard lights.
//
//	<P> Copyright (c) 2006 Blimey! Games Ltd. All rights reserved.
//
//	Author: 
//		Tom Nettleship
//
//	History:
//
//	<TABLE>
//		\Author         Date        Version       Description
//		--------        -----       --------      ------------
//		TNettleship     04/27/2006  0.1           Created
//		TNettleship     05/12/2006  1.0           First "offical" release to FTP
//		TNettleship     05/18/2006  1.1           Added technique annotations.
//		TNettleship     06/09/2006  1.2           Added ambient occlusion term from vertex colour,
//																							global emap factor, removed ambient light, added
//																							option for compressed normals.
//		TNettleship			09/01/2006	1.3						Rewrote the window lighting function.
//		TNettleship			10/03/2006	1.4						Fixed a bug with channel assignment & vertex colours
//																							in 3DSMAX.
//		TNettleship			10/18/2006	1.5						Changed the way lighting distance attenuation works.
//		TNettleship			10/19/2006	1.6						Removed texture mapping info params from display,
//																							reverted to parser 0x0000 until the texture management
//																							bugs in 0x0001 can be fixed.
//		TNettleship			10/25/2006	1.7						Changed binormals from a calculation back to an input.
//		TNettleship			11/02/2006	1.8						Fixed a bug with directional lighting.
//		TMann						11/23/2006	1.10					Added _WIN_GL_ support
//		TMann						11/27/2006	1.11					Texture filtering for GL
//		TMann						12/11/2006	1.13					PS3 version
//		TNettleship			30/04/2007	2.0						Added shader specialisation.
//		TNettleship			07/11/2007	2.01					Changed lighting to work in world coords.
//		TNettleship     07/24/2007  2.02          Made sure samplers aren't using anisotropic filtering.
//		TNettleship     08/17/2007  2.03					Removed texture bias.
//		TNettleship     10/09/2007  2.04					Added support for always-on emission, added centroid interpolation.
//		TNettleship			10/23/2007	2.05					Converted to on-load rendermode behaviour binding.
//	<TABLE>
//
//*************************************************************************************************

#define _SSAO_READY_

#include "stddefs.fxh"
#include "specialisation_globals.fxh"

//-----------------------------------------------------------------------
//
// Preprocessor definitions
//

#define MAX_ENVMAP_BIAS	6.0f



//-----------------------------------------------------------------------
//
// Input parameters
//

//
// Camera
//
#ifdef _3DSMAX_
// 3DSMax parser 0x0001 doesn't support WorldCameraPosition, so we need to bring the view matrix
// in to access the 4th row to get the same information. Parser 0x0000 supports it. Bleh.
float4x4 viewI : ViewInverse
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
// The ingame renderer directly supplies the camera position
SHARE_PARAM float3 worldCameraPos : WorldCameraPosition
<
	string UIWidget = "None";
	bool appEdit = false;
>;
#endif



//
// Transforms
//

#if defined( _3DSMAX_ ) || defined(USE_WVP_CONSTANT)
// Max doesn't support viewproj as an app-supplied parameter
float4x4 worldviewproj : WorldViewProjection
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
SHARE_PARAM float4x4 viewproj : ViewProjection
<
	bool appEdit = false;
	bool export = false;
>;
#endif

float4x4 world : World
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = true;
>;

#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
float4x4 worldI : WorldInverse
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = true;
>;
#endif


//
// Channel mappings (max only)
//

//
// N.B. Max contains a bug which means the colour channel must NOT be mapped to texcoord0.
// The first UV coord channel MUST be mapped to texcoord0 or the basis vectors for normal
// mapping will be screwed up. (e.g. there's some bit of code deep within max which assumes
// this setup when calculating the basis vectors)
//

#ifdef _3DSMAX_

// First UV channel
int texcoord0 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 0;
	int MapChannel = 1;
	int RuntimeTexcoord = 0;
	bool export = false;
> = 0;

// Vertex colour channel
int texcoord1 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 1;
	int MapChannel = 0;
	bool ColorChannel = true;
	bool export = false;
> = 0;

int texcoord2 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 2;
	int MapChannel = 2;
	int RuntimeTexcoord = 1;
	bool export = false;
> = 0;

#endif


//
// Textures
//

#ifdef _3DSMAX_
texture diffuseTexture : DiffuseMap						// Diffuse colour in RGB, window effect scalar in alpha
#else
texture diffuseTexture : TEXTURE							// Diffuse colour in RGB, window effect scalar in alpha
#endif
<
string UIName = "Diffuse Tex {UV1}";
	bool appEdit = true;
>;

texture emissiveTexture : TEXTURE							// Window lighting emissive colour texture
<
string UIName = "Emissive Tex {UV1}";
	bool appEdit = true;
>;

texture specularTexture : TEXTURE							// Specular colour in RGB, shininess in alpha
<
string UIName = "Spec Tex {UV1}";
	bool appEdit = true;
>;

SPECIALISATION_PARAM( useNormalMap, "Use normal map?", "NORMAL_MAPPING" )	// TRUE if the normal map is to be used in lighting
DECLARE_DEPENDENT_VERTEX_STREAM( tangentDependency, tangent, TANGENT, useNormalMap )
DECLARE_DEPENDENT_VERTEX_STREAM( binormalDependency, binormal, BINORMAL, useNormalMap )

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
DEPENDENT_TEXTURE_PARAM( normalTexture, "Normal Tex {UV1}", useNormalMap )
DEPENDENT_BOOL_PARAM( specialNormalMap, "Special normal map format?", useNormalMap )
#endif

SPECIALISATION_PARAM( useEnvironmentMap, "Use environment map?", "ENV_MAPPING" )	// TRUE if the environment map is to be used

#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
DEPENDENT_CUBE_TEXTURE_PARAM( environmentTexture, "Env Texture", useEnvironmentMap, false )
#endif

float minSpecPower
<
	string UIName = "Min Specular Power";
	float UIMin = 1.0f;
	float UIMax = 1024.0f;
	bool appEdit = true;
> = 1.0f;

float maxSpecPower
<
	string UIName = "Max Specular Power";
	float UIMin = 1.0f;
	float UIMax = 1024.0f;
	bool appEdit = true;
> = 32.0f;

float globalSpecularFactor
<
	string UIName = "Specular Factor";
	float UIMin = 0.0f;
	float UIMax = 1.0f;
	bool appEdit = true;
> = 1.0f;

#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( globalEMapFactor, "EMap Factor", useEnvironmentMap, 0.0f, 1.0f, 1.0f )
#endif

#define NUM_EMISSION_CONTROL_VALUES		20
float emissionControl[NUM_EMISSION_CONTROL_VALUES]
<
	string UIName = "emissionControl";
	bool appEdit = true;
	bool export = true;
>; 


// Simplify shadowing code to access only the first (the nearest) shadowmap.
SPECIALISATION_PARAM( forceFistShadowmap, "Force the first shadowmap?", "FORCE_FIRST_SHADOWMAP" )


//
// Lighting
//

#include "lighting_globals.fxh"
DECLARE_LIGHTING_PARAMS

// colour multiplier, forced to end to avoid artists touching it

float4 globalColourMultiplier
<
	string UIWidget = "None";
	bool appEdit = true;
	bool export = true;
> = { 1.0f, 1.0f, 1.0f, 1.0f };


//-----------------------------------------------------------------------
//
// Samplers
//

sampler2D diffuseMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="diffuseTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < diffuseTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};

sampler2D emissiveMap : SAMPLER 
<
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="emissiveTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < emissiveTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};

sampler2D specularMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="specularTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < specularTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
sampler2D normalMap : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="normalTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < normalTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};
#endif

#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
samplerCUBE environmentMap : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="environmentTexture";
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Clamp";
	string AddressV  = "Clamp";
	string AddressW  = "Clamp";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < environmentTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
#if defined(_PS3_)
	AddressU  = ClampToEdge;
	AddressV  = ClampToEdge;
	AddressW  = ClampToEdge;
	LODBias = 0;
#else
	AddressU  = Clamp;
	AddressV  = Clamp;
	AddressW  = Clamp;
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};
#endif


//-----------------------------------------------------------------------
//
// Functions
//

// Function calculates emissiveness by looking up a value from an array which the app sets,
// indexed by the control ID passed. If a negative ID is passed, the emission is considered to
// be always on.

float CalcEmission( float controlID )
{
	int intControlID = floor( controlID );

	if ( intControlID >= 0 )
	{
		return emissionControl[ controlID ];
	}
	else
	{
		return 1.0f;
	}
}

//-----------------------------------------------------------------------
//
// Vertex Shader(s)
//

// Input structure
struct VSINPUT
{
	float3 position  : POSITION;													// Object space position
#ifdef _3DSMAX_
	float4 colour    : TEXCOORD1;													// Vertex colour
	float2 texCoord0 : TEXCOORD0;													// UV channel 1 texture coord - N.B. MAx requires that texcoord0 is a geometric channel
																												// as it implicitly uses that to calculate the tangent space coordinate frame.
	float2 texCoord1 : TEXCOORD2;													// UV channel 2 texture coord
#else
	float4 colour    : COLOR0;														// Vertex colour
	float2 texCoord0 : TEXCOORD0;													// UV channel 1 texture coord
	float2 texCoord1 : TEXCOORD1;													// UV channel 2 texture coord
#endif
	float3 normal    : NORMAL;														// Object space normal


#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
	// These two vertex streams aren't needed without normal_mapping
	float3 tangent   : TANGENT;														// Object space tangent
	float3 binormal	 : BINORMAL;													// Object space binormal
#endif
};


// Output structure
struct VSOUTPUT
{
	float4 position		: POSITION;													// View-coords position
	float4 colour			: TEXCOORD4;														// Vertex colour
	float2 texCoord0	: TEXCOORD0;												// UV coords for texture channel 0
	float  emission		: TEXCOORD1;												// self-illumination
	float3 normal			: TEXCOORD2;												// Normal vector (world space)
	float4 eye				: TEXCOORD3;												// Eye vector (world space)

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
	// These two vertex streams aren't needed without normal_mapping
	float3 tangent		: TEXCOORD5;												// Tangent vector (world space)
	float3 binormal		: TEXCOORD6;												// Binormal vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS_VS( 7 )
#else
	DECLARE_LIGHTING_INTERPOLATORS_VS( 5 )
#endif
};



//-----------------------------------------------------------------------
//
// Vertex shader code
//
// TODO: rearrange to transform eye and lights into local space before
// vector calculations.

VSOUTPUT WindowsVertexShader( VSINPUT _input )
{
	VSOUTPUT _output;

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	// Copy simple invariant params to output structure
	_output.colour = _input.colour;
	_output.texCoord0 = _input.texCoord0;
	_output.emission = CalcEmission( _input.texCoord1.x );

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );

	// Calculate vert's world position
	float3 worldPos = mul( float4( _input.position, 1.0f ), world ).xyz;

	// Calculate world-space vector to the eye
#ifdef _3DSMAX_
// Uncomment if we return to parser 0x0001
	float3 worldEyeVec = viewI[ 3 ] - worldPos;
#else
	float3 worldEyeVec = worldCameraPos - worldPos;
#endif
	_output.eye = float4(worldEyeVec,0);

	// Calculate world-space coordinate frame
	float3 normal = normalize( mul( float4( _input.normal, 0.0f ), world ).xyz );
	_output.normal = normal;

	DEPENDENT_CODE_START( useNormalMap )
#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
		_output.tangent  = mul( float4( _input.tangent, 0.0f ), world ).xyz;
		_output.binormal = mul( float4( _input.binormal, 0.0f ), world ).xyz;
#endif
	DEPENDENT_CODE_END( useNormalMap )

	// Do lighting
	DO_VS_LIGHTING_CALCULATIONS

	_output.colour *= globalColourMultiplier;

	return _output;
}



//-----------------------------------------------------------------------
//
// Fragment Shader(s)
//

#if defined( _3DSMAX_ )
// Max can't handle centroid interpolators properly

// Input structure
struct PSINPUT
{
	float4 colour			: TEXCOORD4;										// Vertex colour
	float2 texCoord0	: TEXCOORD0;								// UV coords for texture channel 0
	float  emission		: TEXCOORD1;								// self-illumination
	float3 normal			: TEXCOORD2;								// Normal vector (world space)
	float4 eye				: TEXCOORD3;								// Eye vector (world space)
	float3 tangent		: TEXCOORD5;								// Tangent vector (world space)
	float3 binormal		: TEXCOORD6;								// Binormal vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS_PS( 7 )
};

#else

struct PSINPUT
{
	float4 colour			: TEXCOORD4;														// Vertex colour
	float2 texCoord0	: TEXCOORD0;												// UV coords for texture channel 0
	float  emission		: TEXCOORD1_centroid;								// self-illumination
	float3 normal			: TEXCOORD2_centroid;								// Normal vector (world space)
	float4 eye				: TEXCOORD3_centroid;								// Eye vector (world space)

#if defined( NORMAL_MAPPING )
	// These two vertex streams aren't needed without normal_mapping
	float3 tangent		: TEXCOORD5_centroid;								// Tangent vector (world space)
	float3 binormal		: TEXCOORD6_centroid;								// Binormal vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS_PS( 7 )
#else
	DECLARE_LIGHTING_INTERPOLATORS_PS( 5 )
#endif
	DECLARE_SHADOW_PS_INPUTS
};

#endif


// Output structure
struct PSOUTPUT
{
	COLOUR_OUTPUT_TYPE Colour : COLOR0;
};



//-----------------------------------------------------------------------
//
// Fragment shader code
//

REMOVE_UNUSED_INTERPOLATORS
PSOUTPUT WindowsFragmentShader( PSINPUT _input )
{
	PSOUTPUT _output;

	PS_GENERATE_WORLDPOS( _input.eye.xyz )

	// Read textures
	float4 diffuseTexColour = tex2D( diffuseMap, _input.texCoord0 );
	float4 specularTexColour = tex2D( specularMap, _input.texCoord0 );
	float4 emissiveColour = tex2D( emissiveMap, _input.texCoord0 );

  // Normalise interpolated vectors
	float3 TSnormal = normalize( _input.normal );
  float3 eye = normalize( _input.eye.xyz );
	float3 normal;

	// Do tangent space normal mapping if required
	DEPENDENT_CODE_START( useNormalMap )
#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
		// Normalise the input tangent and binormal vectors
		float3 tangent = normalize( _input.tangent );
		float3 binormal = normalize( _input.binormal );

		// Fetch and decode the map normal
		float4 normalMapColour = tex2D( normalMap, _input.texCoord0 );
		float3 normalFromMap;

		if ( specialNormalMap )
		{
			normalFromMap.rgb = normalize( ( normalMapColour.agb * 2.0f ) - 1.0f );
		}
		else
		{
			normalFromMap = normalize( ( normalMapColour.rgb * 2.0f ) - 1.0f );
		}

		// Perturb the tangent space normal by the normal map
		normal = ( TSnormal * normalFromMap.z ) + ( normalFromMap.x * binormal ) + ( normalFromMap.y * tangent );
		normal = normalize( normal );
#endif
	DEPENDENT_CODE_ELSE( useNormalMap )
#if defined( _3DSMAX_ ) || !defined( NORMAL_MAPPING )
		// No normal map, so use interpolated normal and constant specular strength
		normal = TSnormal;
#endif
	DEPENDENT_CODE_END( useNormalMap )

	// Calculate base colour
	float4 accumulatedColour = diffuseTexColour * _input.colour;

	// Add in emissive lighting
	accumulatedColour.rgb += _input.emission * emissiveColour.rgb * diffuseTexColour.a;

	// If environment mapping is switched on
	DEPENDENT_CODE_START( useEnvironmentMap )
#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
		// Calculate the reflection vector
		float3 objectSpaceNormal = mul( float4( normal, 0.0f ), worldI ).xyz;
		float3 objectSpaceEye = mul( float4( -eye, 0.0f ), worldI ).xyz;
		float3 reflectionVector = reflect( objectSpaceEye, objectSpaceNormal );

#ifdef _3DSMAX_
		float4 environmentTexColour = texCUBElod( environmentMap, float4( reflectionVector.xzy, MAX_ENVMAP_BIAS * ( 1.0f - specularTexColour.a ) ) );
#else
		float4 environmentTexColour = texCUBElod( environmentMap, float4( reflectionVector, MAX_ENVMAP_BIAS * ( 1.0f - specularTexColour.a ) ) );
#endif

		// Calculate envmap colour and add to ambient and diffuse
		accumulatedColour += specularTexColour * environmentTexColour * globalEMapFactor;

#endif
	DEPENDENT_CODE_END( useEnvironmentMap )

	// Perform lighting
	float	globalSpecularFactorValue = globalSpecularFactor;
	float	minSpecPowerValue = minSpecPower;
	float	maxSpecPowerValue = maxSpecPower;
	DO_PS_LIGHTING_CALCULATIONS( accumulatedColour , _input.eye.xyz )

	accumulatedColour.w = 1.0f;
	_output.Colour = CalculateOutputPixel(accumulatedColour);

	return _output;
}



//-----------------------------------------------------------------------
//
// Technique(s)
//

technique Windows
<
	bool supportsSpecialisedLighting = true;
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "Windows";
	int    normalDeferredID		= 0;
	string zprimeBehaviour		= "ERMB_RENDER_DEFAULT";
	string zprimeDOFBehaviour	= "ERMB_RENDER_DEFAULT";
	string shadowGenBehaviour = "ERMB_RENDER_DEFAULT";
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = true;
		bool AlphaBlendEnable = false;
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = true;
		AlphaBlendEnable = false;
#endif

#if defined (_PS3_)		
		VertexShader = compile sce_vp_rsx WindowsVertexShader();
		PixelShader = compile sce_fp_rsx WindowsFragmentShader();
#else
		VertexShader = compile vs_3_0 WindowsVertexShader();
		PixelShader = compile ps_3_0 WindowsFragmentShader();
#endif		
	}
}
